// water_hlsl2.vsh

uniform mat4 vc0;	// worldViewProj matrix
uniform mat4 vc20;	// worldViewProjScreen
uniform vec4 vc28;	// modelSpaceCamPos
uniform vec3 vc33;	// sunDirection
uniform vec4 vc35;	// texMtxParams
uniform vec3 vc36;	// fakeSunDirection

#define worldViewProj vc0
#define worldViewProjScreen vc20
#define modelSpaceCamPos vc28
#define sunDirection vc33
#define texMtxParams vc35
#define fakeSunDirection vc36

varying vec3 ViewVec;
varying vec3 TanSunDir;
varying vec4 ScreenPos;
varying vec2 FoamCoords;
varying vec3 HalfVector;

void main()
{
	// Transform position
	gl_Position = gl_Vertex * worldViewProj;

	// Set tex coords
#ifdef NOCOLOR
	gl_TexCoord[0].st = gl_MultiTexCoord0.st;
#else
	vec2 temp = gl_MultiTexCoord0.st * texMtxParams.x;
	vec2 bumpCoords;
	bumpCoords.x = temp.x * texMtxParams.w + temp.y * -texMtxParams.z;
	bumpCoords.y = temp.x * texMtxParams.z + temp.y * texMtxParams.w + texMtxParams.y;
	gl_TexCoord[0].st = bumpCoords;
#endif

	// Calc view vector
	vec3 modelSpaceDirToCamera = normalize(modelSpaceCamPos.xyz - gl_Vertex.xyz);
	ViewVec = vec3(-modelSpaceDirToCamera.x, modelSpaceDirToCamera.yz);

	// Get screen space position of vertex
	ScreenPos = gl_Vertex * worldViewProjScreen;

	// Diffuse vertex color
#ifdef NOCOLOR
	gl_FrontColor = vec4(1.0, 1.0, 1.0, 1.0);
#else
	gl_FrontColor = gl_Color;
#endif
   
	// Texture coordinates for foam
#ifdef NOCOLOR
	FoamCoords = gl_MultiTexCoord0.st;
#else
	vec2 foamCoords2;
	vec2 foamCoords = gl_MultiTexCoord0.st * 0.0625;
	foamCoords2.x = foamCoords.x * texMtxParams.w + foamCoords.y * -texMtxParams.z;
	foamCoords2.y = foamCoords.x * texMtxParams.z + foamCoords.y *  texMtxParams.w + texMtxParams.y;
	FoamCoords = foamCoords2;
#endif

	////////////////////////////////////////////////////////////////////////////
	// Move sun direction into tangent space.
	vec3 tangent = vec3(1,0,0);
	vec3 binormal = vec3(0,0,1);
	TanSunDir.x = dot(tangent, sunDirection);
	TanSunDir.y = dot(binormal, sunDirection);
	TanSunDir.z = dot(gl_Normal, sunDirection);

	////////////////////////////////////////////////////////////////////////////
	// Compute H vector using fakeSunDir
	vec3 halfAngleVector = normalize(fakeSunDirection + modelSpaceDirToCamera);

	// Move H vector into tangent space.
	HalfVector.x = dot(tangent, halfAngleVector);
	HalfVector.y = dot(binormal, halfAngleVector);
	HalfVector.z = dot(gl_Normal, halfAngleVector);
}


/*
//-----------------------------------------------------------------------------
// Defines
//-----------------------------------------------------------------------------
float4x4 worldViewProj			: register(c0);
float4x4 worldView				: register(c4);
float4x4 viewT					: register(c8);
float4   colorScales 		: register(c12);
float4x4 worldViewProjScreen	: register(c20);
float2   texCoordScale			: register(c24);
float4	 modelSpaceCamPos		: register(c28);

int4	 ambientColor			: register(c32);
float3	 sunDirection			: register(c33);
int4	 sunColor				: register(c34);
float4	 texMtxParams		: register(c35);
float3	 fakeSunDirection : register(c36);

//-----------------------------------------------------------------------------
// Vertex shader output structure
//-----------------------------------------------------------------------------
struct VS_OUTPUT
{
    float4 Position   : POSITION;   // vertex position 
    float4 Diffuse    : COLOR0;     // vertex diffuse color
    float2 TexCoord0  : TEXCOORD0;  // vertex texture coord 0
    float3 ViewVec    : TEXCOORD1;	// view vector
    float3 TanSunDir  : TEXCOORD2;  // Fresnel factor
    float4 ScreenPos  : TEXCOORD3;  // Screen space project of vertex position
    float2 FoamCoords : TEXCOORD4;  // Foam texture coords
    float3 HalfVector : TEXCOORD5;  // H Vector using fake sun direction
};


//-----------------------------------------------------------------------------
// main
//-----------------------------------------------------------------------------
VS_OUTPUT main( in float4 vPosition  : POSITION,
                in float4 vNormal    : NORMAL,
                in float2 vTexCoord0 : TEXCOORD0,
                in float4 vColor	 : COLOR0 )
{
   VS_OUTPUT Output;

   // Transform position
   Output.Position = mul(vPosition, worldViewProj);

   // Set tex coords
   #ifdef NOCOLOR
      Output.TexCoord0 = vTexCoord0;
   #else
      float2 temp = vTexCoord0 * texMtxParams.x;
      float2 bumpCoords;
      bumpCoords.x = temp.x * texMtxParams.w + temp.y * -texMtxParams.z;
      bumpCoords.y = temp.x * texMtxParams.z + temp.y * texMtxParams.w + texMtxParams.y;
      Output.TexCoord0 = bumpCoords;
   #endif

   // Put view vector in TexCoord1
   float3 modelSpaceDirToCamera = normalize(modelSpaceCamPos.xyz - vPosition.xyz);
   Output.ViewVec = float3(-modelSpaceDirToCamera.x, modelSpaceDirToCamera.yz);

   // Get screen space position of vertex
   Output.ScreenPos = mul(vPosition, worldViewProjScreen);
   
   // Diffuse vertex color
   #ifdef NOCOLOR
      Output.Diffuse = float4(1.0f, 1.0f, 1.0f, 1.0f);
   #else
      Output.Diffuse = vColor;
   #endif
   
   // Texture coordinates for foam
   #ifdef NOCOLOR
      Output.FoamCoords = vTexCoord0;
   #else
      float2 foamCoords2;
      float2 foamCoords = vTexCoord0 * 0.0625f;
      foamCoords2.x = foamCoords.x * texMtxParams.w + foamCoords.y * -texMtxParams.z;//- time * 1;
      foamCoords2.y = foamCoords.x * texMtxParams.z + foamCoords.y *  texMtxParams.w + texMtxParams.y;
      //foamCoords.x += time;
      Output.FoamCoords = foamCoords2;//vPosition.xz * 0.0625f;
   #endif
   
   ////////////////////////////////////////////////////////////////////////////
   // Move sun direction into tangent space.
   float3 tangent = float3(1,0,0);
   float3 binormal = float3(0,0,1);
   Output.TanSunDir.x = dot(tangent.xyz, sunDirection.xyz);
   Output.TanSunDir.y = dot(binormal.xyz, sunDirection.xyz);
   Output.TanSunDir.z = dot(vNormal.xyz, sunDirection.xyz);

   ////////////////////////////////////////////////////////////////////////////
   // Compute H vector using fakeSunDir
   float3 halfAngleVector = normalize(fakeSunDirection.xyz + modelSpaceDirToCamera.xyz);
   
   // Move H vector into tangent space.
   Output.HalfVector.x = dot(tangent.xyz, halfAngleVector.xyz);
   Output.HalfVector.y = dot(binormal.xyz, halfAngleVector.xyz);
   Output.HalfVector.z = dot(vNormal.xyz, halfAngleVector.xyz);

   return Output;
}
*/

